// ロータリーエンコーダー信号のデコード
// Created by Tatsuya Shirai
// National Institute of Technology, Suzuka college
// Mechanical Department
//
// Ver. 1.0 : 2023.05.17
// Ver. 2.0 : 2023.05.23（使用不可）/Crowbar対応
// Ver. 2.1, Ver. 2.2, Ver. 2.3 : 2023.05.24：繰り返し実行，移動平均機能追加，信号チェック機能追加
// Ver. 2.4.0 : 2023.05.24 CDS（角速度）計算周辺の機能強化
// Ver. 2.5.0 : 2023.05.30 acc（累積値）をintからfloatに変更
// Ver. 2.6.0 : 2023.06.02 解析結果の抜き取り機能の追加
// Ver. 2.7.0 : 2023.06.05 経過時間の列の追加（出力データ）

// グローバル変数（システム管理）
boolean waitingProcess;  // ファイルオープン後，処理の終了を待つ
File    selected;        // fileselector()で開かれたファイルのポインター（未選択状態はnull）

// 変数初期化
void variableInitialize() {
  waitingProcess = false;
  selected       = null;
}

void Setup()
{
  variableInitialize();
  
  // 画面位置を変更する
  surface.setLocation(Init_PosX, Init_PosY);

  programComment("ロータリーエンコーダーのデコード（Ver." + Version + ")");
  crowbar.disableTextArea();
  parameter("中心差分近似（角速度）の前後データ距離：n >= 1").setInt(CDS).label("CDS");
  parameter("パルス累積の移動平均（前後n項：n=0の時は移動平均しない").setInt(MApulse).label("MApulse");
  parameter("角速度の移動平均（前後n項：n=0の時は移動平均しない").setInt(MAav).label("MAav");
  parameter("計算結果の抜き取り間隔（*.txtファイルを出力／n=0の時は抜き取らない").setInt(Extract).label("Extract");
  parameter("閾値High").setFloat(SH_H).label("SH_H");
  parameter("閾値Low"). setFloat(SH_L).label("SH_L");
  parameter("CW/CCWを反転しますか？(0:false, 1:true)").setBoolean(REVERSE).label("REVERSE");
  parameter("サンプリング間隔[ms]").setFloat(DeltaT).label("DeltaT");
  parameter("ロータリーエンコーダーのカウント数").setInt(EncCount).label("EncCount");
//drop_init();
}

void Main()
{
  SH_H     = crowbar.getFloat("SH_H");
  SH_L     = crowbar.getFloat("SH_L");
  REVERSE  = crowbar.getBoolean("REVERSE");
  CDS      = crowbar.getInt("CDS");
  DeltaT   = crowbar.getFloat("DeltaT");
  EncCount = crowbar.getInt("EncCount");
  MApulse  = crowbar.getInt("MApulse");
  MAav     = crowbar.getInt("MAav");
  Extract  = crowbar.getInt("Extract");

  display_notice();
  // 入力されたパラメーターの値域のチェック
  if (CDS <= 0) {
    msg.setError("CDSの値は1以上を指定して下さい：" + CDS);
  } else if (DeltaT <= 0.0) {
    msg.setError("サンプリング間隔の値は0より大きな値を指定して下さい：" + DeltaT);
  } else if (EncCount <= 0) {
    msg.setError("ロータリーエンコーダーのカウント数は0より大きな値を指定して下さい：" + EncCount);
  } else if ((MApulse < 0) || (MAav < 0)) {
    msg.setError("移動平均の項数は0以上の値を指定して下さい（パルス累積：角速度）：（" + MApulse + "：" + MAav + "）");
  } else if (Extract < 0) {
    msg.setError("抜き取り間隔数は0以上の値を指定してください：" + Extract);
  } else {
    return;  // 正常時
  }
  crowbar.stop();  // 異常時
}

// 注意の表示
void display_notice() {
  int tx = 10, ty = 10;
  int dy = 32;

  crowbar.clrscr();
  crowbar.textColor(#000000);
  writeln("【ファイル形式】");
  writeln("・カンマ区切りのCSVファイル");
  writeln("・見出し行なし（１行目からデータ）");
  writeln("・空行無し");
  writeln("・データ１,データ２");
  writeln("　（３列目以降なし）");
  writeln("・ファイル選択キャンセル or ESC or Q で終了");
  writeln("・出力ファイルは無条件で上書きされますので注意して下さい");
  writeln("・出力ファイルが別のアプリケーションで開かれていると異常終了します");
  writeln("・データ冒頭と末尾CDS個分の移動平均(pps)のデータは不正確です");
  // メッセージの表示
  crowbar.textColor(#ff0000);
  writeln("CSVファイルを選択して下さい");
  crowbar.textColor(#000000);
  writeln("");
}

Message msg = new Message();

// 読み込み可能なデータ形式（カンマ区切りのCSV）
// 一行目からデータがある（見出しは削除して下さい）
// 一行に少なくとも２データある
// データの無い行を発見したらエラー終了

// ファイルの入力関係
class ImportData {
  String fullpathname;  // 読み込むCSVファイル
  String filename;      // ファイル名の部分（拡張子含む）
  
  float   [] Af;        // A相の電圧（勿論，0/1でも構わない）
  float   [] Bf;        // B相の電圧（勿論，0/1でも構わない）
  int        max_num;   // データ数
  String  [] lines;     // 読み込んだテキストデータ

  // コンストラクター
  ImportData() {
    max_num = 0;
  }

  // ファイル形式が正しいかどうかをチェック
  boolean validation() {
    CsvStrings csv;

    for (int i = 0; i < max_num; i++) {
      csv = new CsvStrings(lines[i]);
      // ２列のデータが浮動小数点数かチェック
      for (int j = 0; j < 2; j++) {
        if (! csv.next()) {
          msg.setError(i + "行目のデータの" + (j + 1) + "列目のデータが存在しません");
          return false;
        }
        if (! csv.is_float()) {
          msg.setError(i + "行目のデータの" + (j + 1) + "列目のデータが浮動小数点数ではありません");
          return false;
        }
      }
      if (csv.next()) {
        msg.setError(i + "行目のデータに３列目以降のデータが存在します");
        return false;
      }
    }
    return true;
  }

  // 選択されたファイルを読み込む
  boolean readData(File F) {
    CsvStrings csv;      // CSVデータを分割するクラス

    // ファイルの情報を読み出す
    fullpathname = F.getAbsolutePath();
    filename     = F.getName();
    // ファイルを読み込む
    if ((lines = loadStrings(fullpathname)) == null) {
      msg.setError("指定されたファイルは読み込めません");
      return false;
    }
    msg.set(filename + " を読み込みます");
    // 一行も読み込めなかった
    if ((max_num = lines.length) == 0) {
      msg.setError("データがありません（テキストファイルではないのかも知れません）");
      return false;
    }
    // 読み込んだファイルのチェック
    if (! validation()) {
      msg.setError("ファイル形式が正しくありません");
      return false;
    }
    msg.set(max_num + "行のデータを読み込みます");
    // 生データの読み込み（電圧）
    // データ領域を確保
    Af  = new float[max_num];
    Bf  = new float[max_num];
    // データ領域に読み込む
    for (int i = 0; i < max_num; i++) {
      csv = new CsvStrings(lines[i]);
      csv.next();
      Af[i] = csv.value;
      csv.next();
      Bf[i] = csv.value;
    }
    return true;
  }
}

ImportData im_data = new ImportData(); 
import java.util.*;

import java.io.*;
String convert2SJIS(String str) {
  String ret="";
  try{
    byte[] bytes=str.getBytes("SJIS");
    ret = new String(bytes,"SHIFT_JIS");
  }catch(UnsupportedEncodingException e){
    //変換失敗時の処理適宜
    println("文字コード変換エラー");
  }
  return ret;
}

import java.math.BigDecimal;

// ファイル出力関係
class ExportFile {
  String      dest_filename;
  PrintWriter output;
  boolean exporting;
  
  ExportFile() {
    dest_filename = "";
    output        = null;
    exporting     = false;
  }
  // ファイルへ出力
  // ext = trueの時、抜き取り出力
  boolean export_data(boolean ext) {
    exporting = true;  // 例外処理用フラグ
    dest_filename = im_data.fullpathname + ".csv";  // 拡張子の分解は面倒くさいので，単純に拡張子を追加する
    if (ext && (Extract > 0)) dest_filename += ".txt";       // 抜き取り出力時は更に".txt"を追加する
    output = createWriter(dest_filename);
    exporting = false;
    if (output == null) {
      msg.setError("ファイルへの保存に失敗：" + dest_filename);
      return false;
    }
    String headerString, dataString;
    // ファイルヘッダーの出力
    headerString  = im_data.filename + ",";
    headerString += year() + "/" + nf(month(),2) + "/"  + nf(day(), 2) + "," + nf(hour(), 2) + ":" + nf(minute(), 2) + ":" + nf(second(), 2);
    output.println(headerString);
    // パラメーターの出力
    headerString = "SH_H,SH_L,REVERS,CDS,DeltaT,EncCount,MApulse,MAav";
    output.println(headerString);
    headerString = SH_H + "," + SH_L + "," + REVERSE + "," + CDS + "," + DeltaT + "," + EncCount + "," + MApulse + "," + MAav;
    output.println(headerString);
    // 見出し行の出力
    headerString = "No,Time, A-Phase(raw),B-Phase(raw),A-Phase(01),B-Phase(01),CW/CCW,Accumulation,Round,Differential(pps),AngularVelocity(rpm),AngularVelocity(rad/s)";
    headerString = convert2SJIS(headerString);
    output.println(headerString);

    // データ本体の出力
    int cnt  = 0;  // 通しのカウンター
    int skip = 0;  // Extract > 0（抜き取り有効）の時の読み飛ばし用カウンター
    // 時間の計算に誤差が生じるため
    BigDecimal bd1 = new BigDecimal(nf(DeltaT));
    BigDecimal bd2 = new BigDecimal("1000");
    BigDecimal bd3 = bd1.divide(bd2);
    for (cnt = 0; cnt < data.max_num; cnt++) {
      // 抜き取り出力時のスキップ動作
      if (ext && (Extract > 0)) {
        if (++skip < Extract) {
          continue;
        }
        skip = 0;  // 出力時：カウンターをゼロに戻す
      }
      dataString  = (cnt + 1) + ",";                 // Ver.2.6.0から追加
      bd1 = bd3.multiply(new BigDecimal(nf(cnt)));
      dataString += nf(bd1.floatValue()) + ",";      // Ver.2.7.0から追加
      dataString += nf(data.Af[cnt]) + "," + nf(data.Bf[cnt]) + ",";
      dataString += (data.A[cnt] ? "1" : "0") + "," + (data.B[cnt] ? "1" : "0") + ",";
      dataString += data.enc[cnt] + ",";
      dataString += data.acc[cnt] + ",";
      dataString += data.acc[cnt] / (EncCount * 4.0) + ",";  // Ver.2.7.1から追加
      dataString += data.pps[cnt] + "," + data.rpm[cnt] + "," + data.radps[cnt];
      output.println(dataString);
    }
    output.flush();
    output.close();
    if (! ext) msg.set("解析結果を" + dest_filename + "に保存しました");
      else     msg.set("抜き取りを" + dest_filename + "に保存しました");
    return true;
  }
}

ExportFile ex_file = new ExportFile();

//// ファイル・フォルダー選択ダイアログボックス関係の関数
// ファイル選択ダイアログボックスの呼び出し
void open_fileselector()
{
  selectInput("Select CSV file", "fileSelected");
}
// ファイル選択後の処理
void fileSelected(File selection) 
{
  selected = selection;
  if (selected == null) {
    msg.set("ファイル選択がキャンセルされました");
    waitingProcess = false;
    crowbar.stop();
    return;
  }
} //<>//
